%% REPLACEMENT_NSGA2
function new_population = replacement_nsga2(population, old_population, ...
                                            fitness_functions, param)
% Replacement using NSGA-II
% NEW_POPULATION = REPLACEMENT_NSGA2(POPULATION, OLD_POPULATION,
% FITNESS_FUNCTIONS, PARAM) New population is created using NSGA-II. The max
% depth of the first front and the depth distributions are recorded. The
% production choice unique IDs are counted.
%    
%    POPULATION cell array of new population
%    OLD_POPULATION cell array of old population
%    FITNESS_FUNCTIONS array of fitness functions
%    PARAM struct with parameters

max_depth_front_one = param.CFG.min_depth;
%Row 1 is the front depths. Row 2 is the cumulative sum of the occurances
%on the depths in the front
depths_front_one = [max_depth_front_one; 1];
total_puids = [];
depths_front_one = [];
production_choice_id_range = 1:param.CFG.unique_production_choice_ids{1, ...
                    end}{end};

%% Sort into fronts
tot_population = [population; old_population];
[f_fronts, fitness_functions] = non_dominated_sort(fitness_functions, tot_population);
new_population = [];
i = 1;
while size(new_population,1) < param.POPULATION_SIZE
    % Setting the rank to the front the individuals belong to
    pop_f = tot_population(f_fronts(i).f,:);    
    for j = 1:size(pop_f, 1)
        pop_f(j).RANK_C = i;
    end
    %% Calculate crowding distance
    pop_f = calculate_crowding_distance(fitness_functions(f_fronts(i).f,:), ...
                                        pop_f, param);
    new_population = [new_population; pop_f];
    %% Max depth and depth distribution of rank 1
    if i == 1
        max_depth_front_one = max([pop_f.DEPTH_C]);
        edges = min([pop_f.DEPTH_C]):1:max([pop_f.DEPTH_C]);
        [n depth_bin] = histc([pop_f.DEPTH_C], edges);
        depth_bin_cum = cumsum(n);
        depths_front_one = [edges; depth_bin_cum];
    end
    %% Production choice ID distribution
    pop_puids = [pop_f.PRODUCTION_CHOICE_IDS_C];
    pop_puids = reshape(pop_puids, size(pop_puids,1) * size(pop_puids,2), 1);
    n_puids = histc(pop_puids, production_choice_id_range);
    total_puids = [total_puids; n_puids'./size(pop_f,1)];
    i = i + 1;
end

%% Print total PUIDs
for i = 1:size(total_puids,1)
    str_ = sprintf('PUIDs %d', i);
    for j = 1:size(total_puids,2)
        str_ = [str_ sprintf(' %3.3f', total_puids(i,j))];
    end
    fprintf(param.FID,'%s\n', str_);
end

%% Sort population
new_population_mat = [new_population.RANK_C; new_population.DISTANCE_C]';
[new_population_mat index] = sortrows(new_population_mat, [1 2]);
new_population = new_population(index);
new_population = new_population(1:param.POPULATION_SIZE);

%% NON_DOMINATED_SORT
function [f_fronts fitness_functions] = non_dominated_sort(fitness_functions, population)
% Sort the current popultion based on non-domination.
    
% F_FRONTS = NON_DOMINATED_SORT(FITNESS_FUNCTIONS, POPULATION) Sort the
% current popultion based on non-domination. All the individuals in the
% first front are given rank 1, the second front rank 2... After assigning
% rank the crowding for each front is calculated.  *Kalyanmoy Deb, Amrit
% Pratap, Sameer Agarwal, and T. Meyarivan*, |A Fast Elitist Multiobjective
% Genetic Algorithm: NSGA-II|, IEEE Transactions on Evolutionary Computation
% 6 (2002), no. 2, 182 ~ 197.

% FITNESS_FUNCTIONS is matirx with fitness functions on each column and the
% individuals fitness on each row 
% POPULATION is a cell array, population of individual solutions

f_fronts.f = [];

individual = population;

for i = 1 : size(individual,1)
    % Number of individuals that dominate this individual
    individual(i).n = 0;
    % Individuals which this individual dominate
    individual(i).p = [];
    for j = 1 : size(individual, 1)
        dom_less = 0;
        dom_equal = 0;
        dom_more = 0;
        %Go throught the fitness functions
        for k = 1 : size(fitness_functions,2)
            if (fitness_functions(i,k) > fitness_functions(j,k))
                dom_less = dom_less + 1;
            elseif (fitness_functions(i,k) == fitness_functions(j,k))
                dom_equal = dom_equal + 1;
            else
                dom_more = dom_more + 1;
            end
        end
        %If j dominates i
        if dom_less == 0 && dom_equal ~= size(fitness_functions, 2)
            %Increase domination counter of i
            individual(i).n = individual(i).n + 1;
        elseif dom_more == 0 && dom_equal ~= size(fitness_functions, 2)
            %Add j to individuals dominated by i
            individual(i).p = [individual(i).p j];
        end
    end
    %i belongs to the first front if it is not dominated
    if individual(i).n == 0
        f_fronts(1).f = [f_fronts(1).f i];
    end
end
%for i = 1:size(individual,1)
%   fprintf(param.FID,'%d %d %d %.3f %.3f %.3f\n', i, individual(i).n, length(individual(i).p), individual(i).c3);
%end
% Find the subsequent fronts
% Initialize front counter
front = 1;
while ~isempty(f_fronts(front).f)
    % Store members of the next front
    Q = [];
    for i = 1:length(f_fronts(front).f)
        if ~isempty(individual(f_fronts(front).f(i)).p)
            for j = 1:length(individual(f_fronts(front).f(i)).p)
                individual(individual(f_fronts(front).f(i)).p(j)).n = individual(individual(f_fronts(front).f(i)).p(j)).n - 1;
                % j belongs to the next front
                if individual(individual(f_fronts(front).f(i)).p(j)).n == 0
                    Q = [Q individual(f_fronts(front).f(i)).p(j)];
                end
            end
        end
    end
    front = front + 1;
    f_fronts(front).f = Q;
end

%% CALCULATE_CROWDING_DISTANCE
function individuals = calculate_crowding_distance(fitness_functions, ...
                                                   individuals, param)
% Calculates the distance between solutions, how crowded they are

% INDIVIDUALS = CALCULATE_CROWDING_DISTANCE(FITNESS_FUNCTIONS, INDIVIDUALS)
% The distance between INDIVIDUALS based on the FITNESS_FUNCTIONS is
% calculated and set, as for the NSGA-II. The fitness function values of the
% individuals are assumed to be normalized, [0,1]
    
% FITNESS_FUNCTIONS the fitness_functions
% INDIVIDUALS the solutions considered
% PARAM struct of parameters
    
%Set distance to 0
for i = 1:size(individuals, 1)
    individuals(i).DISTANCE_C = 0;
end

% Assuming that min is 0 and max is 1, since normalizing
f_min = zeros(1,param.FITNESS_FUNCTIONS);
f_max = ones(1,param.FITNESS_FUNCTIONS);

% Number of solutions in I
for i = 1:size(individuals,1)
    for j = 1:size(fitness_functions, 2)
        % Sort using each objective value
        [temp,index_of_fronts] = sort(fitness_functions(:,j));
        % Make sure boundary values are always selected
        individuals(index_of_fronts(1)).DISTANCE_C = inf;
        individuals(index_of_fronts(end)).DISTANCE_C = inf;
        for k = 2:size(temp,1) - 1
            individuals(index_of_fronts(k)).DISTANCE_C = ...
                individuals(index_of_fronts(k)).DISTANCE_C + ...
                ( (temp(k+1) - temp(k-1)) / (f_max(j) - f_min(j)) );
        end
    end
end

% Negating distance since rank is sorted in descending order
for i = 1:size(individuals, 1)
    individuals(i).DISTANCE_C = -individuals(i).DISTANCE_C; 
end
