function gema(params)
% GEM Grammatical Evolution in Matlab (GEM)
% GEMA(PARAMS) implements the Evolutionary Computaiton method
% Grammatical Evolution and uses PARAMS for parameter settings. 

% TODO update function documentation
% TODO reduce number of for loops, use cellfun?
% TODO store output as a string all the time??
% TODO remove defaults and force crash for missing parameters
% TODO save and load all individual data, not only genotypes
% TODO change PARISIMONY_PREASSURE to PARISMONY_PRESSURE
% TODO fix population saving
% TODO print param to file
% TODO plot average fitness for multiobjective
    
%% Get arguments
param = struct();
if nargin == 0
    %Default GEM
    addpath('symbolic_regression');
    %addpath('financial_modelling');
    addpath('operators');
    param = symbolic_regression_config(param);
    %param = financial_modelling_config(param);
else
    param = params;
end


%% Set variables

param.MAX_INVALID_REMAPS = 1000; %Avoid invalids
param.MAX_SIMULATION_RESTARTS = 10; %Avoid similar power outputs
param.IND_SIZE = length(fieldnames(param.INDIVIDUAL_STRUCT));
if strcmp(param.ORDER,'ascend')
    param.MIN_FITNESS = -1 * param.MIN_FITNESS;
end

param.FID = 1;
if isfield(param, 'SAVE_DIRECTORY_PATH')
    if exist(param.SAVE_DIRECTORY_PATH, 'dir')
        out_file = strcat(param.SAVE_DIRECTORY_PATH,'/gema_',num2str(param.RUN),'.out');
        param.FID =fopen(out_file,'w');
    else
       exception = MException('gema:FileNotFound',sprintf('File not found:%s', out_file));
       throw(exception); 
    end
end
disp('Used param:');
disp(param);

% Values for random number generator
param.SEED = sum(100*clock);
if verLessThan('matlab', '7.6')
    fprintf(param.FID,'Version %s using my_randi for %random integers\n', version('-release'))
    param.RANDINT = @my_randi;
    %Not using randn
    rand('twister', param.SEED);
else
    param.RANDINT = @randi;
    RandStream.setDefaultStream(RandStream('mt19937ar','seed',param.SEED));
    param.DEFAULT_STREAM = RandStream.getDefaultStream;
    disp(param.DEFAULT_STREAM);
end

% Stats structure keeps run statistics
stats.fitness_evaluation_cnt = 0;
stats.mapping_cnt = 0;
% Checking a valid phenotype but not calling the fitness function
stats.expression_check_cnt = 0;
% Store expressions and fitnesses that have already been calculated 
stats.fitness_store = {};

%% Read grammar file
fid = fopen(param.GRAMMAR_FILE,'r');
lines = {};
cnt = 1;
while 1
    line = fgetl(fid);
    if ~ischar(line)
        break
    end
    lines(cnt) = cellstr(line);
    cnt = cnt + 1;
end
fclose(fid);
lines = strtrim(lines);

%% Parse BNF lines to grammar
param.CFG = get_grammar(lines);

%% Setup multicore, using the multicore package
if isfield(param,'MULTICORE')
    settings.multicoreDir = param.MULTICORE_DIR;
    settings.nrOfEvalsAtOnce = param.NEVALS_AT_ONCE; 
    settings.maxEvalTimeSingle = param.FEVALS_TIMES_SINGLE * 2;
    settings.masterIsWorker = true; % default: true
    settings.useWaitbar = false;
else
    settings = struct();
end

%% Initialize population. 

% Load a population. Read from file if param.LOAD_POPULATION exists. The
% saved population is the genotypes.
inputs = cell(param.POPULATION_SIZE, 1);
if isfield(param, 'LOAD_POPULATION')
    fid = fopen(param.LOAD_POPULATION, 'r');
    i = 1;
    line = fgetl(fid);
    while ischar(line)
        inputs{i,1} = str2num(line);
        line = fgetl(fid);
        i = i + 1;
    end
    fclose(fid);
    if (i - 1) ~= param.POPULATION_SIZE
        error('Not matching population size of current run(%d) and loaded population(%d)', param.POPULATION_SIZE, (i - 1));
    end
    fprintf(param.FID,'Loading population from %s\n', param.LOAD_POPULATION);
else
    % Create the inputs
    % TODO if ramped half half is used avoid mapping again
    inputs = param.INITIALISATION(param.POPULATION_SIZE, param);
end

pop(param.POPULATION_SIZE, 1) = param.INDIVIDUAL_STRUCT;
% Create the individuals
for i = 1:1:size(inputs,1)    
    pop(i) = create_individual(cell2mat(inputs(i,:)), param);
    %% Map the individual.
    % Set the input(genotype), output(phenotype), used input, depth and
    % production choices. The input can be changed by map_individual
    % to avoid invalid individuals
    [output used_input input depth production_choice_ids, stats] = ...
        map_individual(pop(i).INPUT_C, 0, param, stats);
    pop(i).INPUT_C = input;
    pop(i).OUTPUT_C = sprintf('%s', output{:});
    pop(i).USED_INPUT_C = used_input;
    pop(i).DEPTH_C = depth;
    pop(i).PRODUCTION_CHOICE_IDS_C = production_choice_ids;
end

%% Initial fitness evaluation. Generation is 0
[pop stats] = param.FITNESS_EVALUATION_METHOD(pop, settings, 0, param, stats);

%% Sort by rank, NSGA-II population is already sorted from the replacement
% TODO better indicator of multiple fitnesses than IND_SIZE, e.g size(param.FITNESS_C)
if param.IND_SIZE == 6
    fitnesses = [pop.FITNESS_C];
    [fitnesses fitness_index] = sort(fitnesses, 2, param.ORDER);
    pop = pop(fitness_index, :);
end

%% Create strings for printing individuals
best_ind_str = '%s,';
ind_str = 'IND,%5d,%4d,%4d,';
for i = 1:param.FITNESS_FUNCTIONS
    best_ind_str = strcat(best_ind_str, '%.3f,');
    ind_str = strcat(ind_str, '%7.3f,');
end
best_ind_str = strcat(best_ind_str, '%3d,%3d\n');
ind_str = strcat(ind_str, '%3d,%3d');
if param.IND_SIZE > 6
    ind_str = strcat(ind_str, ',', '%3d,%7.3f');
end
ind_str = strcat(ind_str, '\n');

%% Start evolutionary iterations
for gen = 1:1:param.GENERATIONS
    tic    
    %% Print population stats
    out = '';
    if ~isempty(pop(1).OUTPUT_C)
        out = sprintf('%s', pop(1).OUTPUT_C);
    end
    inputs_ind = zeros(size(pop,1),1);
    for i = 1:size(pop,1)
        inputs_ind(i) = size(pop(i).INPUT_C,2);
    end
    pop_gen = param.POPULATION_SIZE * gen;
    gen_data = [gen mean([pop.USED_INPUT_C]) ...
                std([pop.USED_INPUT_C]) mean([pop.DEPTH_C]) ...
                std([pop.DEPTH_C]) stats.fitness_evaluation_cnt ...
                stats.mapping_cnt stats.expression_check_cnt ...
                size(stats.fitness_store,2) mean([pop.FITNESS_C]) ...
                std([pop.FITNESS_C]) mean(inputs_ind) ...
                std(inputs_ind)];
    gen_data(6:9) = gen_data(6:9)./pop_gen;
    % Print population values, i.e mean and std. Used input and depth
    fprintf(param.FID,'ITR,%d,%.3f,%.3f,%.3f,%.3f,%d,%d,%d,%d,', gen_data(1), ...
            gen_data(2), gen_data(3), gen_data(4), gen_data(5), ...
            gen_data(6), gen_data(7), ...
            gen_data(8), gen_data(9));
    % Print best individual 
    fprintf(param.FID,best_ind_str, out, pop(1).FITNESS_C,...
            pop(1).USED_INPUT_C, pop(1).DEPTH_C);
    
    % All the fields should be there if FITNESS_PLOT is there
    if isfield(param, 'FITNESS_PLOT') 
        plot_gui(param, pop, gen_data, out, inputs_ind);
    end

    %% Select the new population
    new_pop = param.SELECTION(param, pop);
        
    %%Crossover
    xo_pop = param.CROSSOVER_OPERATION(new_pop, param);
    
    %% Mutation
    for i = 1:1:size(xo_pop,1)
        c_ind = xo_pop(i,:);
        xo_pop(i) = param.MUTATION_OPERATION(c_ind, param);
        
        %% Map the individual.
        % Set the input(genotype), output(phenotype), used input, depth and
        % production choices. The input can be changed by map_individual
        % to avoid invalid individuals
        [output used_input input depth production_choice_ids, stats] = ...
            map_individual(xo_pop(i).INPUT_C, gen, param, stats);
        xo_pop(i).INPUT_C = input;
        xo_pop(i).OUTPUT_C = sprintf('%s', output{:});
        xo_pop(i).USED_INPUT_C = used_input;
        xo_pop(i).DEPTH_C = depth;
        xo_pop(i).PRODUCTION_CHOICE_IDS_C = production_choice_ids;
    end
    
    %% Fitness evaluation
    [fe_pop stats] = param.FITNESS_EVALUATION_METHOD(xo_pop, settings, gen, param, stats);
    
    %% Replacement
    if param.IND_SIZE == 6
        pop = param.REPLACEMENT(pop, fe_pop, param);
    else
        % Reshape the fitness for replacement
        fitness_functions = [pop.FITNESS_C fe_pop.FITNESS_C];
        fitness_functions_2 = reshape(fitness_functions, ...
                                      param.FITNESS_FUNCTIONS, size(pop,1) + size(fe_pop,1))';
        pop = param.REPLACEMENT(pop, fe_pop, fitness_functions_2, param);
    end

    %% Population statiscs
    if param.IND_SIZE == 6
        fitnesses = [pop.FITNESS_C];
        [fitnesses fitness_index] = sort(fitnesses, 2, param.ORDER);
        pop = pop(fitness_index, :);
        for ind = 1:size(pop,1)
            ind_data = [ind, size(pop(ind).INPUT_C,2), size(pop(ind).OUTPUT_C,2), ...
                        pop(ind).FITNESS_C, pop(ind).USED_INPUT_C, pop(ind).DEPTH_C];
            fprintf(param.FID,ind_str, ind_data);
        end        
    else
        for ind = 1:size(pop,1)
            ind_data = [ind, size(pop(ind).INPUT_C,2), size(pop(ind).OUTPUT_C,2), ...
                        pop(ind).FITNESS_C, pop(ind).USED_INPUT_C, ...
                        pop(ind).DEPTH_C, pop(ind).RANK_C, pop(ind).DISTANCE_C];
            fprintf(param.FID,ind_str, ind_data);
        end        
    end
    toc
end

%% Save population
if isfield(param, 'SAVE_POPULATION')
    save_file = fopen(param.SAVE_POPULATION, 'w');
    for i = 1:1:size(pop, 1)          
        fprintf(save_file, '%s\n', num2str(pop(i).INPUT_C));
    end
    fclose(save_file);
end

%% Print population stats
out = '';
if ~isempty(pop(1).OUTPUT_C)
    out = sprintf('%s', pop(1).OUTPUT_C);
end
% Print population values, i.e mean and std. Used input and depth
fprintf(param.FID,'END %d %.3f %5.3f %.3f %.3f ', gen, ...
        mean([pop.USED_INPUT_C]), std([pop.USED_INPUT_C]), ...
        mean([pop.DEPTH_C]), std([pop.DEPTH_C]));
% Print best individual 
fprintf(param.FID,best_ind_str, out, pop(1).FITNESS_C,...
        pop(1).USED_INPUT_C, pop(1).DEPTH_C);

fprintf(param.FID,'END fitness evaluations %d extra mappings: %d extra expression checks: %d extra fitness_evaluations: %d fitness_store: %d\n', ...
        (param.POPULATION_SIZE * param.GENERATIONS), ...
        stats.mapping_cnt, stats.expression_check_cnt, ...
        stats.fitness_evaluation_cnt, size(stats.fitness_store,2));
if param.FID ~= 1
    fclose(param.FID);
end
%% MY_RANDI
function data = my_randi(max_value, rows, columns)
% Wrapper function for randi when using Matlab < 8.0
    
% DATA = MY_RANDI(MAX_VALUE, ROWS, COLUMNS) returns DATA a matrix or
% scalar of uniformly distributed random integer.
% MAX_VALUE integer max malue
% ROWS integer number of rows of matrix
% COLUMNS integer number of columns of matrix
    
data = ceil(rand(rows, columns) * max_value);    
    
%% PLOT_GUI
function plot_gui(param, pop, gen_data, out, inputs_ind)
% Plot on the GUI
    
set(param.BEST_INDIVIDUAL, 'String', out);
axes(param.FITNESS_PLOT);
hold on;
if size(pop(1).FITNESS_C,2) > 1
    plot(gen_data(1), pop(1).FITNESS_C,'+');
    %        errorbar(gen_data(1), gen_data(10), gen_data(11), 'o');
    %plot(gen_data(1), gen_data(10), 'or');
    legend_str = {};
    for i = 1: size(pop(1).FITNESS_C,2)
        legend_str = [legend_str, strcat('Best Fit. obj.', num2str(i))];
    end
    legend(legend_str);
else
    plot(gen_data(1), mean(pop(1).FITNESS_C),'--sb');
    %        errorbar(gen_data(1), gen_data(10), gen_data(11), 'o');
    plot(gen_data(1), gen_data(10), 'or');
    legend('Best Fit.', 'Ave. Fit.');
end    

axes(param.OTHER_PLOT);
hold on;
plot(gen_data(1), pop(1).USED_INPUT_C,'--sr');
errorbar(gen_data(1), gen_data(2), gen_data(3), 'o');
plot(gen_data(1), inputs_ind(1),'--.b');
errorbar(gen_data(1), gen_data(12), gen_data(13), '+');
legend('Best Ind. Used Input', 'Ave. Used Input','Best Ind. Input Size', 'Ave. Input Size');

axes(param.OTHER_PLOT_1);
hold on;
plot(gen_data(1), gen_data(7),'--or');
legend('Mapping Calls','Location','SouthEast');
